/*!
 * Copyright (c) 2005, Freescale Semiconductor
 *
 * Freescale Confidential Proprietary
 * \file    mcu_spi_config.c
 * \brief   This is the SMAC C source for the SPI driver
 * \author  Larry Roshak
 * \version 4.1a
 * \date    2005/08/11 20:44:37 
 * 
 * \b Description:
 *
 * This is the SMAC C source driver file for the 
 * The driver control interface to the SPI.
 *
 * \b Department: Freescale Radio Products Division 
 *
 * \b Project: SMAC (Simple Media Access Controller)
 *
 * \b History:
 * - 17/04/2004 : Initial Development. (FLR005)
 * - 07/01/2005 : Code Standarization. (A19259)
 * - 11/16/2005 Doc. update to Doxygen compliant by Gonzalo Delgado 
 * Huitron rgd04c
 */ 

#include "hprfgw_config.h"
#if defined (OS_WINCE)
 #include <windows.h>
 #include <types.h>
 #include <ceddk.h>
 #include <ddkreg.h>
 #include <serhw.h>
 #include "ecspibus.h"
 #include <bsp.h>
 #include "hprf_winceIntfc.h"
#endif
#include "smac_pub_def.h"
#include "smac_mcu_hw_config.h"
#include "smac_drivers.h"
#include "smac_MC13192_hw_config.h"
#include "smac_MC13192_regs.h"
#include "smac_mcu_spi_config.h"
#include "hprfgw_rfSlaveIntToHost.h"
#if defined (OS_NUCLEUS)
 #include "mcf5271.h"
 #include "system.h"
#endif

// We use Mode 0 (POL bit = 0 (polarity active high), PHA bit = 0 (phase 0, latch on odd edge) 
// Similar to what we used in Nautilus
/*
    0               0       Mode 0 spin_clk active high and sampling occurs on the
                            rising edge.
    0               1       Mode 1 spin_clk active high and sampling occurs on the
                            falling edge.
    1               0       Mode 2 spin_clk active low and sampling occurs on the
                            falling edge.
    1               1       Mode 3 spin_clk active low and sampling occurs on the
                            rising edge.
    Frequency       6Mhz    PHY can support upto 8Mhz, let's choose 6Mhz
    Chip Select     SS1     CS1 of ECSPI1
    Word Len        8 bits  Refered as burst length here (MSB will be shifted out
                            first)
    References              Page 389 of iMX53 reference manual for Input selection
                            for the pins.
                            Page 1043 of the manual for ECSPI chapter.
*/

////////////////////////////////////////////////////////////////
// ECSPI configurations and functions - Start
//////////////////////////
static CSPI_BUSCONFIG_T BusCfgLocal =
{
    HPR_SPI_CS,							// ChannelSelect
    HPR_SPI_FREQ,						// Freq
    8,									// BurstLength
    CSPI_CONFIGREG_SSBPOL_ACTIVELOW,	// SSPOL   - active low signal to the chip
    CSPI_CONFIGREG_SCLKPOL_ACTIVEHIGH,  // SCLKPOL - clock will be low when not active
    CSPI_CONFIGREG_SCLKPHA_PHASE0,      // SCLKPHA - phase 0 (data will be latched on the odd edge of clock)
    CSPI_CONTROLREG_DRCTL_DONTCARE,     // DRCTL (Don't care DATA_READY Ctrl bit, not used in hardware)
    FALSE,                              // usedma
};

static CRITICAL_SECTION g_csECSPI_CriticalSection;
static volatile PCSP_ECSPI_REG g_pECSPI = NULL;

// Global Variables
static volatile PCSP_EPIT_REG g_pEPIT = NULL;

// GPIO port in the SPI's Chip select pin in present and the GPIO bit
#define SPI_CS_GPIO_PORT                DDK_GPIO_PORT3
#define SPI_CS_GPIO_PIN                 19
#define SPI_BASE_ADDRESS                CSP_BASE_REG_PA_ECSPI1      /* We use ECSPI1 for Radio */
#define SPI_CLK_GPIO_PORT               DDK_GPIO_PORT3
#define SPI_CLK_GPIO_PIN                16
#define SPI_MOSI_GPIO_PORT              DDK_GPIO_PORT3
#define SPI_MOSI_GPIO_PIN               18



// Assert Chip select
void EnableChannel()
{
    DDKGpioWriteDataPin(SPI_CS_GPIO_PORT, SPI_CS_GPIO_PIN, 0);
}

// Deassert chip select
void DisableChannel()
{
    DDKGpioWriteDataPin(SPI_CS_GPIO_PORT, SPI_CS_GPIO_PIN, 1);
}

// Enable ECSPI hardware
void EnableECSPI()
{
    // Reinitialize the pins again
	//ECSPI1_SCLK
    DDKIomuxSetPinMux(DDK_IOMUX_PIN_EIM_D16, DDK_IOMUX_PIN_MUXMODE_ALT4, DDK_IOMUX_PIN_SION_REGULAR);
    // Select the daisy (page 389 of iMX53 reference manual
    DDKIomuxSelectInput(DDK_IOMUX_SELECT_INPUT_ECSPI1_IPP_CSPI_CLK_IN, 3);

    //ECSPI1_MOSI
    DDKIomuxSetPinMux(DDK_IOMUX_PIN_EIM_D18, DDK_IOMUX_PIN_MUXMODE_ALT4, DDK_IOMUX_PIN_SION_REGULAR);
    // Select the daisy (page 389 of iMX53 reference manual
	DDKIomuxSelectInput(DDK_IOMUX_SELECT_INPUT_ECSPI1_IPP_IND_MOSI, 3);
	
	DDKClockSetGatingMode(DDK_CLOCK_GATE_INDEX_ECSPI1_IPG, DDK_CLOCK_GATE_MODE_ENABLED_ALL);
    DDKClockSetGatingMode(DDK_CLOCK_GATE_INDEX_ECSPI1_PERCLK, DDK_CLOCK_GATE_MODE_ENABLED_ALL);

    // Enable CSPI to reset internal logic
    INSREG32BF(&g_pECSPI->CONTROLREG, CSPI_CONTROLREG_EN, CSPI_CONTROLREG_EN_ENABLE);
}

// Disable ECSPI hardware
void DisableECSPI()
{
    // disable the CSPI
    INSREG32BF(&g_pECSPI->CONTROLREG, CSPI_CONTROLREG_EN, CSPI_CONTROLREG_EN_DISABLE);

    DDKClockSetGatingMode(DDK_CLOCK_GATE_INDEX_ECSPI1_IPG, DDK_CLOCK_GATE_MODE_DISABLED);
    DDKClockSetGatingMode(DDK_CLOCK_GATE_INDEX_ECSPI1_PERCLK, DDK_CLOCK_GATE_MODE_DISABLED);

	// Drive the interface signals low - CS1
	DDKGpioWriteDataPin(SPI_CS_GPIO_PORT, SPI_CS_GPIO_PIN, 0);

	//ECSPI1_SCLK - GPIO3_16
	DDKGpioWriteDataPin(SPI_CLK_GPIO_PORT, SPI_CLK_GPIO_PIN, 0);
	DDKIomuxSetPinMux(DDK_IOMUX_PIN_EIM_D16, DDK_IOMUX_PIN_MUXMODE_ALT1, DDK_IOMUX_PIN_SION_REGULAR);
	DDKGpioSetConfig(SPI_CLK_GPIO_PORT, SPI_CLK_GPIO_PIN, DDK_GPIO_DIR_OUT, DDK_GPIO_INTR_NONE);

	//ECSPI1_MOSI - GPIO3_18
	DDKGpioWriteDataPin(SPI_MOSI_GPIO_PORT, SPI_MOSI_GPIO_PIN, 0);
	DDKIomuxSetPinMux(DDK_IOMUX_PIN_EIM_D18, DDK_IOMUX_PIN_MUXMODE_ALT1, DDK_IOMUX_PIN_SION_REGULAR);
	DDKGpioSetConfig(SPI_MOSI_GPIO_PORT, SPI_MOSI_GPIO_PIN, DDK_GPIO_DIR_OUT, DDK_GPIO_INTR_NONE);
}

// Write and read through ECSPI
DWORD SPIWriteRead(DWORD dwCount, UINT8* txBuf, UINT8* rxBuf)
{
    DWORD dwByteCount = dwCount;
    UINT8 *pTx, *pRx;
    DWORD dwWrite, dwRead;
	PHYSICAL_ADDRESS phyAddr;
	DWORD initialTime,elapsedTime;
	const DWORD dwTimeout = 8*20;// 20us at 8MHZ is about 160 ticks
	DWORD dwTemp;
	BOOL bDone;


	 // Initialize timer register for timeout
    if (g_pEPIT == NULL)
    {
    	phyAddr.QuadPart = CSP_BASE_REG_PA_EPIT1;
        
        // Map peripheral physical address to virtual address
        g_pEPIT = (PCSP_EPIT_REG) MmMapIoSpace(phyAddr, sizeof(PCSP_EPIT_REG), FALSE);

    }

    /* Using a debug pin of the processor to toggle during SPI
    transfer, it is found that:
    Time taken to do one transaction (3 bytes write) is about 12uS
    Time taken to send about 50 byte packet is about 150uS
    
    Used the SD3_WP (EIM_D12) pin which comes at the SD card slot J9
    on the PCB, which is the right most pin on the bottom, by holding
    the panel such that the microSD card slot is at upper right corner
    to the SD card slot and the text J9 is read in right way. */

    pTx = &txBuf[0]; 
    pRx = &rxBuf[0];

    EnterCriticalSection(&g_csECSPI_CriticalSection);
    
    // Assert CE (this can be moved to any other place if needed.
    EnableChannel();

    while (dwByteCount)
    {

        dwWrite = (ULONG)*pTx;
        
        // Write data
        OUTREG32(&g_pECSPI->TXDATA, dwWrite);

		/* There is a problem with some (but not all) of the TST-600 panels where the XCH bit
		(SPI Exchange Bit) in the CONREG does not clear at the end of a transfer. This causes
		a timeout error which does not stop the transfer, but makes it take much longer than
		it should and burns up CPU cycles while waiting. In order to work around this problem,
		we will look at both the XCH bit and the TC (Transfer Complete) bit in the STATREG.
		Both of these bits should indicate that a transfer has run to completion. In order to
		use the TC bit, it must be cleared by writing a "1" to it before each transfer.

		Using only the TC bit greatly reduced the problem, but did not completely eliminate it.
		Thus, we are using both bits.
		*/

		// clear "transfer complete" bit in status register
		dwTemp = INREG32(&g_pECSPI->STATREG);
		dwTemp |= 1U << CSPI_STATREG_TC_LSH;
		dwTemp &= ~(1U << CSPI_STATREG_RO_LSH);
        OUTREG32(&g_pECSPI->STATREG, dwTemp);

        // start exchange
        INSREG32BF(&g_pECSPI->CONTROLREG, CSPI_CONTROLREG_XCH, CSPI_CONTROLREG_XCH_EN);

		initialTime = g_pEPIT->CNT;
		elapsedTime = 0;
		bDone = FALSE;
        // Wait for transfer to complete
        while (!bDone &&  (elapsedTime<dwTimeout))
		{
			elapsedTime = 	initialTime - g_pEPIT->CNT;
			if(INREG32(&g_pECSPI->STATREG) & (1U<<CSPI_STATREG_TC_LSH)) bDone = TRUE;
			if(!(INREG32(&g_pECSPI->CONTROLREG) & CSP_BITFMASK(CSPI_CONTROLREG_XCH))) bDone = TRUE;
		}
        if ( !bDone )
        {
            RFIntHostApp.pF_ErrorPrintf("SPIWriteRead(): Transfer Time Out!\r\n");
        }

        /* Note that we still need to wait for the receive bit before we read otherwise. According to the reference manual
        page no 1071 (application flow chart) it is sufficient to wait for the XCH bit but it is observed that the XCH bit
        was found cleared sometimes the very first time we check above after setting it. If we read at that time, we end up
        getting wrong data (too early reading). Since we know that every transfer out, we get data in, so we will wait */

		initialTime = g_pEPIT->CNT;
		elapsedTime = 0;
		bDone = FALSE;

        // Wait for receive to complete
		while (!bDone &&  (elapsedTime<dwTimeout))
		{
			elapsedTime = 	initialTime - g_pEPIT->CNT;	
			if(INREG32(&g_pECSPI->STATREG) & CSP_BITFMASK(CSPI_STATREG_RR)) bDone = TRUE;
		}

        if ( bDone )
        {
            dwRead = INREG32(&g_pECSPI->RXDATA);
            *pRx = (UINT8)(dwRead & 0x000000FF);
        }
        else
        {
            RFIntHostApp.pF_ErrorPrintf("SPIWriteRead(): Read timeout!\r\n");
        }

        pTx++;
        pRx++;

        dwByteCount--;
    }

    // Deassert CE
    DisableChannel();
    LeaveCriticalSection(&g_csECSPI_CriticalSection);

	return (dwCount-dwByteCount);
}

// Calculate clock rate
void ECSPICalculateDivRate(UINT32 dwFrequency, UINT32 dwTolerance, UINT8 *PREDIV, UINT8 *POSTDIV )
{
    UINT32 dwCspiClk;
    UINT32 tmp=0;

    DDKClockGetFreq(DDK_CLOCK_SIGNAL_ECSPI, &dwCspiClk);
  
    tmp = dwCspiClk/dwFrequency;
    tmp = (dwCspiClk<(tmp*(dwFrequency+dwTolerance)))?(tmp-1):tmp;
    while( tmp >=16 )
    {
        *POSTDIV+=1;
        tmp >>=1;
    }

    *PREDIV = (UINT8)tmp;

    return ;
}
/////////////////////////////////////
// ECSPI configurations and functions - End
/////////////////////////////////////////////////////////////////////


/*!
  \fn void SMAC_SPIInit(void)
  \brief Initialize the SPI Control and Baud Rate Registers
*/
void SMAC_SPIInit(void)
{

    UINT8 u8ChannelSelect;
    UINT8 u8SSPOL;
    UINT8 u8POL;
    UINT8 u8PHA;
    UINT8 u8DRCTL;
    UINT8 PreDiv = 0, PostDiv = 0;
    UINT8 dwSSPOL;

    //ECSPI1_SCLK
    DDKIomuxSetPinMux(DDK_IOMUX_PIN_EIM_D16, DDK_IOMUX_PIN_MUXMODE_ALT4, DDK_IOMUX_PIN_SION_REGULAR);
    DDKIomuxSetPadConfig(DDK_IOMUX_PAD_EIM_D16, DDK_IOMUX_PAD_SLEW_SLOW, DDK_IOMUX_PAD_DRIVE_HIGH,
                         DDK_IOMUX_PAD_OPENDRAIN_DISABLE, DDK_IOMUX_PAD_PULL_NONE, DDK_IOMUX_PAD_HYSTERESIS_ENABLE,
                         DDK_IOMUX_PAD_VDOEN_NULL, DDK_IOMUX_PAD_OUTVOLT_NULL);
    // Select the daisy (page 389 of iMX53 reference manual
    DDKIomuxSelectInput(DDK_IOMUX_SELECT_INPUT_ECSPI1_IPP_CSPI_CLK_IN, 3);


    //ECSPI1_MISO
    DDKIomuxSetPinMux(DDK_IOMUX_PIN_EIM_D17, DDK_IOMUX_PIN_MUXMODE_ALT4, DDK_IOMUX_PIN_SION_REGULAR);
    DDKIomuxSetPadConfig(DDK_IOMUX_PAD_EIM_D17, DDK_IOMUX_PAD_SLEW_SLOW, DDK_IOMUX_PAD_DRIVE_HIGH,
                         DDK_IOMUX_PAD_OPENDRAIN_DISABLE, DDK_IOMUX_PAD_PULL_NONE, DDK_IOMUX_PAD_HYSTERESIS_ENABLE,
                         DDK_IOMUX_PAD_VDOEN_NULL, DDK_IOMUX_PAD_OUTVOLT_NULL);
    // Select the daisy (page 389 of iMX53 reference manual
	DDKIomuxSelectInput(DDK_IOMUX_SELECT_INPUT_ECSPI1_IPP_IND_MISO, 3);

    //ECSPI1_MOSI
    DDKIomuxSetPinMux(DDK_IOMUX_PIN_EIM_D18, DDK_IOMUX_PIN_MUXMODE_ALT4, DDK_IOMUX_PIN_SION_REGULAR);
    DDKIomuxSetPadConfig(DDK_IOMUX_PAD_EIM_D18, DDK_IOMUX_PAD_SLEW_SLOW, DDK_IOMUX_PAD_DRIVE_HIGH,
                         DDK_IOMUX_PAD_OPENDRAIN_DISABLE, DDK_IOMUX_PAD_PULL_NONE, DDK_IOMUX_PAD_HYSTERESIS_ENABLE,
                         DDK_IOMUX_PAD_VDOEN_NULL, DDK_IOMUX_PAD_OUTVOLT_NULL);
    // Select the daisy (page 389 of iMX53 reference manual
	DDKIomuxSelectInput(DDK_IOMUX_SELECT_INPUT_ECSPI1_IPP_IND_MOSI, 3);

    /* We are going to use the CS as a GPIO port - so don't use this alternate mode */
    /*ECSPI1_SS1 (HPR_SPI_CS) */
    /*DDKIomuxSetPinMux(DDK_IOMUX_PIN_EIM_D19, DDK_IOMUX_PIN_MUXMODE_ALT4, DDK_IOMUX_PIN_SION_REGULAR);
    DDKIomuxSetPadConfig(DDK_IOMUX_PAD_EIM_D19, DDK_IOMUX_PAD_SLEW_SLOW, DDK_IOMUX_PAD_DRIVE_HIGH,
                         DDK_IOMUX_PAD_OPENDRAIN_DISABLE, DDK_IOMUX_PAD_PULL_NONE, DDK_IOMUX_PAD_HYSTERESIS_ENABLE,
                         DDK_IOMUX_PAD_VDOEN_NULL, DDK_IOMUX_PAD_OUTVOLT_NULL);
    // Select the daisy (page 389 of iMX53 reference manual
	DDKIomuxSelectInput(DDK_IOMUX_SELECT_INPUT_ECSPI1_IPP_IND_SS_B_1, 2);*/

    //ECSPI1_SS1 (HPR_SPI_CS)
    DDKGpioWriteDataPin(SPI_CS_GPIO_PORT, SPI_CS_GPIO_PIN, 1);
    DDKIomuxSetPinMux(DDK_IOMUX_PIN_EIM_D19, DDK_IOMUX_PIN_MUXMODE_ALT1, DDK_IOMUX_PIN_SION_REGULAR);
    DDKIomuxSetPadConfig(DDK_IOMUX_PAD_EIM_D19, DDK_IOMUX_PAD_SLEW_SLOW, DDK_IOMUX_PAD_DRIVE_HIGH,
                         DDK_IOMUX_PAD_OPENDRAIN_DISABLE, DDK_IOMUX_PAD_PULL_NONE, DDK_IOMUX_PAD_HYSTERESIS_ENABLE,
                         DDK_IOMUX_PAD_VDOEN_NULL, DDK_IOMUX_PAD_OUTVOLT_NULL);
    DDKGpioSetConfig(SPI_CS_GPIO_PORT, SPI_CS_GPIO_PIN, DDK_GPIO_DIR_OUT, DDK_GPIO_INTR_NONE);

    // Initialize everything
    if (g_pECSPI == NULL)
    {
    	PHYSICAL_ADDRESS phyAddr;
	    phyAddr.QuadPart = SPI_BASE_ADDRESS;
        
        // Map peripheral physical address to virtual address
        g_pECSPI = (PCSP_ECSPI_REG) MmMapIoSpace(phyAddr, sizeof(CSP_ECSPI_REG), FALSE);

    }

    InitializeCriticalSection(&g_csECSPI_CriticalSection);

    EnterCriticalSection(&g_csECSPI_CriticalSection);

    // Enable clock gating
    DDKClockSetGatingMode(DDK_CLOCK_GATE_INDEX_ECSPI1_IPG, DDK_CLOCK_GATE_MODE_ENABLED_ALL);
    DDKClockSetGatingMode(DDK_CLOCK_GATE_INDEX_ECSPI1_PERCLK, DDK_CLOCK_GATE_MODE_ENABLED_ALL);

    // disable all interrupts
    OUTREG32(&g_pECSPI->INTREG, 0);

    u8ChannelSelect = BusCfgLocal.ChannelSelect &((1U<<CSPI_CONTROLREG_CHANNELSELECT_WID) -1);
    u8SSPOL = BusCfgLocal.SSPOL? CSPI_CONFIGREG_SSBPOL_ACTIVEHIGH: CSPI_CONFIGREG_SSBPOL_ACTIVELOW;
    u8POL = BusCfgLocal.SCLKPOL? CSPI_CONFIGREG_SCLKPOL_ACTIVELOW: CSPI_CONFIGREG_SCLKPOL_ACTIVEHIGH;
    u8PHA = BusCfgLocal.SCLKPHA? CSPI_CONFIGREG_SCLKPHA_PHASE1: CSPI_CONFIGREG_SCLKPHA_PHASE0;
    u8DRCTL = BusCfgLocal.DRCTL&((1U<<CSPI_CONTROLREG_DRCTL_WID) -1);

    ECSPICalculateDivRate(BusCfgLocal.Freq, BusCfgLocal.Freq/10, &PreDiv,  &PostDiv);

    // ENGcm09397
    dwSSPOL &= ~(1<<u8ChannelSelect);
    dwSSPOL |= u8SSPOL<<u8ChannelSelect;
    // ENGcm09397 end

    // Enable SPI and set all the parameters
    OUTREG32(&g_pECSPI->CONTROLREG, 
        CSP_BITFVAL(CSPI_CONTROLREG_EN, CSPI_CONTROLREG_EN_ENABLE)              |
        CSP_BITFVAL(CSPI_CONTROLREG_HW, CSPI_CONTROLREG_HW_HTMODE_DISABLE)      |
        CSP_BITFVAL(CSPI_CONTROLREG_XCH, CSPI_CONTROLREG_XCH_IDLE)              |
        CSP_BITFVAL(CSPI_CONTROLREG_SMC, CSPI_CONTROLREG_SMC_NORMAL_MODE)       |
        CSP_BITFVAL(CSPI_CONTROLREG_CHANNELMODE, CSPI_CONTROLREG_CHANNELMODE_MASTER<<u8ChannelSelect)|
        CSP_BITFVAL(CSPI_CONTROLREG_PREDIVIDER, PreDiv)                         |
        CSP_BITFVAL(CSPI_CONTROLREG_POSTDIVIDER, PostDiv)                       |
        CSP_BITFVAL(CSPI_CONTROLREG_DRCTL, u8DRCTL)                             |
        CSP_BITFVAL(CSPI_CONTROLREG_CHANNELSELECT, u8ChannelSelect)             |
        CSP_BITFVAL(CSPI_CONTROLREG_BURSTLENGTH, BusCfgLocal.BurstLength-1));

    // Note: SSBCTRL  bit is changed to MULTIPLEBURSTS (compared to what was in actual ECSPI driver)
    OUTREG32(&g_pECSPI->CONFIGREG,
        CSP_BITFVAL(CSPI_CONFIGREG_SCLKPHA, u8PHA<<u8ChannelSelect)   |
        CSP_BITFVAL(CSPI_CONFIGREG_SCLKPOL, u8POL<<u8ChannelSelect)   |
        CSP_BITFVAL(CSPI_CONFIGREG_SSBCTRL, CSPI_CONFIGREG_SSBCTRL_MULTIPLEBURSTS) |
        CSP_BITFVAL(CSPI_CONFIGREG_SSBPOL,  dwSSPOL)                            |//byParamSSPOL<<byParamChannelSelect)  |
        CSP_BITFVAL(CSPI_CONFIGREG_DATACTL, CSPI_CONFIGREG_DATACTL_STAYHIGH)    |
        CSP_BITFVAL(CSPI_CONFIGREG_SCLKCTL, CSPI_CONFIGREG_SCLKCTL_STAYHIGH)    |
        CSP_BITFVAL(CSPI_CONFIGREG_HTLENGTH, 0x0));

    // No loopback
    INSREG32BF(&g_pECSPI->TESTREG, CSPI_TESTREG_LBC, CSPI_TESTREG_LBC_NOTCONNECTED);

    LeaveCriticalSection(&g_csECSPI_CriticalSection);
}


/*!
 * \brief SPIDrvWrite : Write 1 word to SPI
 *
 * \param u8Addr - SPI address
 * \param u16Content - Data to send
 *
 */
void SPIDrvWrite(UINT8 addr, UINT16 content)
{
    UINT8 content_byte[3];

    /*disable interrupts to avoid double access to zigbee spi*/
    MC13192DisableInterrupts();

    content_byte[0] = addr;
    content_byte[1] = (content>>8)&0xFF;
    content_byte[2] = (content)&0xFF;

    SPIWriteRead(3, content_byte, content_byte);

    /*enable interrupts*/
    MC13192RestoreInterrupts();
}


/*!
 * \brief SPIDrvRead : Read 1 word from SPI
 *
 * \param u8Addr - SPI address
 *
 * \return u16Data -  u16Data[0] is the MSB, u16Data[1] is the LSB
 */
UINT16 SPIDrvRead(UINT8 addr)
{
    UINT8 content_byte[3];
    UINT16 content;

    /*disable interrupts to avoid double access to zigbee spi*/
    MC13192DisableInterrupts();

    // Start by writing the address and then read the data (high byte first)
    addr |= 0x80;	// For read, the MSB must be HIGH for the PHY chip. See data sheet, page 66
    content_byte[0] = addr;
    content_byte[1] = 0;		// For tx data
    content_byte[2] = 0;		// For tx data

    SPIWriteRead(3, content_byte, content_byte);

    content = content_byte[2] | (content_byte[1] <<8);

    /*enable interrupts*/
    MC13192RestoreInterrupts();

    return content;
}


/*!
 * \brief RAMDrvWriteTx : Write a block of data to TX packet RAM, whichever is selected
 *
 * \param *psTxPkt - Packet to write
 *
 */
void RAMDrvWriteTx(tTxPacket *tx_pkt)
{
    UINT8 tx_buf[132], u8addr;		//SMAC_PAYLOAD_MAXSIZE 127 +  a few extra bytes
    UINT32 dwTxBytes, dwBufIndex;
    UINT16 tmp;

    /*Prepare the packet length by adding 2 to the total value*/ 
    /*for MDR purposes*/
    tmp = SPIDrvRead(TX_PKT_LEN);
    // MNT - 4/27/2007 - The Header will now be put in by the MAC layer above this
    // We only add for the CRC
    SPIDrvWrite(TX_PKT_LEN,((tmp & 0xFF80) | (tx_pkt->u8DataLength + 2) ));

    // Now disable interrupt
    MC13192DisableInterrupts();

    dwBufIndex = 1;		// Start filling from index 1, Index 0 is for address
    
    // Fill up the buffer
    for (dwTxBytes = 0; dwTxBytes <= tx_pkt->u8DataLength;)
    {
        tx_buf[dwBufIndex] = tx_pkt->pu8Data[dwTxBytes+1];
        tx_buf[dwBufIndex+1] = tx_pkt->pu8Data[dwTxBytes];
        dwTxBytes = dwTxBytes + 2;
        dwBufIndex = dwBufIndex + 2;
    }
    
    u8addr = TX_PKT;	// address of TX_PKT RAM
    tx_buf[0] = u8addr;	// Fill the address in index 0

    // Include address also in the count
    SPIWriteRead((dwTxBytes + 1), tx_buf, tx_buf);

    // Restore interrupts
    MC13192RestoreInterrupts();
}


/*!
 * \brief RAMDrvReadRx : Read a block of data from RX packet RAM, whichever is selected
 *
 * \param *psRxPkt - Packet received
 *
 * \return u8Status
 */
UINT8 RAMDrvReadRx(tRxPacket *rx_pkt)
{
    UINT32 i, ReadBytes;
    UINT8 addr, readbuffer[132 + 8];	// Keep extra 4 bytes + 8 for safety just in case!
    UINT8 *ptrReadBuf;					// Pointer to read buffer

    // Clear the read buffer
    memset(readbuffer, 0, sizeof (readbuffer));

    // We are going to discard first 2 bytes, so subtract 2
    rx_pkt->u8DataLength  = rx_pkt->u8DataLength  - 2;

    /*disable interrupts to avoid double access to zigbee spi*/
    MC13192DisableInterrupts();

    addr = (RX_PKT | 0x80);	// Fill the address and READ bit in the first byte to Tx

    // Adjust the no of bytes to read to even.
    ReadBytes = rx_pkt->u8DataLength;
    if ((ReadBytes % 2))
        ReadBytes++;
    else
        ReadBytes += 2;

    readbuffer[0] = addr;

    // Include address, 2 CRCs also in the count
    SPIWriteRead((ReadBytes + 1 + 2), readbuffer, readbuffer);

    // We have to discard first three bytes what we read, address byte write + 2 CRC bytes
    // So we use a pointer to read buffer.
    ptrReadBuf = &readbuffer[3];

    // Now copy data from buffer to actual buffer by swapping alternate bytes
    // The MC13192 sends bytes in reverse order - BYTE2 first, BYTE1 second.
    for (i = 0; i <= rx_pkt->u8DataLength;)
    {
        if(i+1 == rx_pkt->u8DataLength )
        {
            /*do nothing garbage byte*/
            /*Last Odd byte*/
        }
        else
        {
            rx_pkt->pu8Data[i+1] = ptrReadBuf[i];
        }
        rx_pkt->pu8Data[i] = ptrReadBuf[i+1];
        i = i + 2;
    }
    /*enable interrupts*/
    MC13192RestoreInterrupts();

    return SMAC_SUCCESS;
}

/* This function is not used. Can be used for test purpose */
/*!
 * \brief RAMDrvReadRx : Read a block of data from RX packet RAM, whichever is selected
 *
 * \param *psRxPkt - Packet received
 *
 * \return u8Status
 */
UINT8 RAMDrvReadTx(UINT8 *tx_pkt, UINT16 numbytes)
{

    UINT32 i;
    UINT8 addr, readbuffer[132 + 8];	// Keep extra 4 bytes + 8 for safety just in case!
    UINT8 *ptrReadBuf;					// Pointer to read buffer

    // Clear the read buffer
    memset(readbuffer, 0, sizeof (readbuffer));

    /*disable interrupts to avoid double access to zigbee spi*/
    MC13192DisableInterrupts();

    // Start by writing the address and then read the data (high byte first)
    addr = (TX_PKT | 0x80);	// 0x02 address of Tx packet RAM in the MC13192, MSB = 1 indicates Read

    // load the address to read at index 0 of the buffer
    readbuffer[0] = addr;

    SPIWriteRead((numbytes + 1), readbuffer, readbuffer);

    // Ignore the first byte (since we wrote the address)
    ptrReadBuf = readbuffer[1];

    /*enable interrupts*/
    MC13192RestoreInterrupts();

    // Now copy data from buffer to actual buffer by swapping alternate bytes
    // The MC13192 sends bytes in reverse order - BYTE2 first, BYTE1 second.
    for (i = 0; i < numbytes;)
    {
        if(i+1 == numbytes )
        {
            /*do nothing garbage byte*/
            /*Last Odd byte*/
        }
        else
        {
            tx_pkt[i+1] = ptrReadBuf[i];
        }
        tx_pkt[i] = ptrReadBuf[i+1];
        i = i + 2;
    }
    return SMAC_SUCCESS;
}